<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Doctor;
use App\Notifications\DoctorRegistrationNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class DoctorRegistrationController extends Controller
{
    public function showRegistration()
    {
        return view('doctor.register');
    }

    public function register(Request $request)
    {
        $validated = $request->validate([
            'title' => 'nullable|string',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'specialization' => 'nullable|string',
            'qualification' => 'nullable|string',
            'practice_number' => 'required|string|unique:doctors,practice_number',
            'country' => 'nullable|string',
            'educational_details' => 'nullable|string',
            'phone' => 'nullable|string',
            'email' => 'required|email|unique:users,email|unique:doctors,email',
            'password' => 'required|string|min:8|confirmed',
            'profile_picture' => 'nullable|image|max:2048',
            'letterhead' => 'nullable|image|max:2048',
            'stamp' => 'nullable|image|max:2048',
            'signature' => 'nullable|image|max:2048',
        ]);

        // Create user
        $user = User::create([
            'name' => "{$validated['first_name']} {$validated['last_name']}",
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'doctor',
            'is_active' => false,
        ]);

        // Handle file uploads
        $profilePicturePath = null;
        $letterheadPath = null;
        $stampPath = null;
        $signaturePath = null;

        if ($request->hasFile('profile_picture')) {
            $profilePicturePath = $request->file('profile_picture')->store('doctors/profile-pictures', 'public');
        }

        if ($request->hasFile('letterhead')) {
            $letterheadPath = $request->file('letterhead')->store('doctors/letterheads', 'public');
        }

        if ($request->hasFile('stamp')) {
            $stampPath = $request->file('stamp')->store('doctors/stamps', 'public');
        }

        if ($request->hasFile('signature')) {
            $signaturePath = $request->file('signature')->store('doctors/signatures', 'public');
        }

        // Create doctor profile
        $doctor = Doctor::create([
            'user_id' => $user->id,
            'title' => $validated['title'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'specialization' => $validated['specialization'],
            'qualification' => $validated['qualification'],
            'practice_number' => $validated['practice_number'],
            'country' => $validated['country'] ?? 'Lesotho',
            'educational_details' => $validated['educational_details'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'profile_picture_path' => $profilePicturePath,
            'letterhead_path' => $letterheadPath,
            'stamp_path' => $stampPath,
            'signature_path' => $signaturePath,
            'is_verified' => false,
            'is_active' => false,
        ]);

        // Notify all admins
        $admins = User::where('role', 'admin')->where('is_active', true)->get();
        foreach ($admins as $admin) {
            try {
                $admin->notify(new DoctorRegistrationNotification($doctor));
            } catch (\Exception $e) {
                // Log error but continue - database notification will still work
                \Log::warning('Failed to send email notification to admin: ' . $e->getMessage());
            }
        }

        return redirect()->route('login')
            ->with('success', 'Registration successful! Your account is pending verification by admin.');
    }
}

